import math
import gym
from gym import spaces, logger
from gym.utils import seeding
import numpy as np

class ContinuousCartPoleEnv(gym.Env):
    metadata = {
        "render.modes": ["human", "rgb_array"],
        "video.frames_per_second": 50,
    }

    def __init__(
        self,
        dynamics_noise_scale=0.01,  # standard dev for noise added to velocities
        obs_noise_scale=0.0,        # standard dev for noise added to returned observation
        action_noise_scale=0.0      # standard dev for noise added to the force
    ):
        """
        :param dynamics_noise_scale:   float, std dev for noise to x_dot, theta_dot
        :param obs_noise_scale:        float, std dev for noise to the final observation
        :param action_noise_scale:     float, std dev for noise added to the force
        """
        # self.gravity = 9.8
        self.gravity = 20
        self.masscart = 1.0
        self.masspole = 0.1
        self.total_mass = (self.masspole + self.masscart)
        self.length = 0.5  # actually half the pole's length
        self.polemass_length = (self.masspole * self.length)
        self.force_mag = 30.0
        self.tau = 0.02  # seconds between state updates
        self.min_action = -1.0
        self.max_action = 1.0

        # Stochastic parameters
        self.dynamics_noise_scale = dynamics_noise_scale
        self.obs_noise_scale = obs_noise_scale
        self.action_noise_scale = action_noise_scale

        # Angle at which to fail the episode
        self.theta_threshold_radians = 12 * 2 * math.pi / 360
        self.x_threshold = 2.4

        # Angle limit set to 2 * theta_threshold_radians, so the failing
        # observation is still within bounds
        high = np.array(
            [
                self.x_threshold * 2,
                np.finfo(np.float32).max,
                self.theta_threshold_radians * 2,
                np.finfo(np.float32).max,
            ],
            dtype=np.float32
        )

        self.action_space = spaces.Box(
            low=self.min_action,
            high=self.max_action,
            shape=(1,),
            dtype=np.float32
        )
        self.observation_space = spaces.Box(
            -high, high, dtype=np.float32
        )

        self.seed()
        self.viewer = None
        self.state = None
        self.steps_beyond_done = None

    def seed(self, seed=None):
        self.np_random, seed = seeding.np_random(seed)
        return [seed]

    def stepPhysics(self, force):
        """
        Compute the next state of the cart-pole given current state and
        a (possibly noisy) force.
        """
        x, x_dot, theta, theta_dot = self.state

        costheta = math.cos(theta)
        sintheta = math.sin(theta)
        temp = (force + self.polemass_length * theta_dot * theta_dot * sintheta) / self.total_mass
        thetaacc = (self.gravity * sintheta - costheta * temp) / (
            self.length * (4.0 / 3.0 - self.masspole * costheta * costheta / self.total_mass)
        )
        xacc = temp - self.polemass_length * thetaacc * costheta / self.total_mass

        # Update state
        x = x + self.tau * x_dot
        x_dot = x_dot + self.tau * xacc
        theta = theta + self.tau * theta_dot
        theta_dot = theta_dot + self.tau * thetaacc

        # Inject stochasticity in the dynamics: Add small noise to the velocities
        x_dot += self.np_random.normal(0.0, self.dynamics_noise_scale)
        theta_dot += self.np_random.normal(0.0, self.dynamics_noise_scale)

        return (x, x_dot, theta, theta_dot)

    def step(self, action):
        # Enforce that action is in the space
        assert self.action_space.contains(action), f"{action} invalid"

        # Possibly add action noise
        raw_force = float(action[0])
        noisy_force = raw_force + self.np_random.normal(0.0, self.action_noise_scale)
        # Clip the final force to keep it within the environment's intended range if desired
        noisy_force = np.clip(noisy_force, self.min_action, self.max_action)
        # Scale to actual force
        force = self.force_mag * noisy_force

        # Update the state
        self.state = self.stepPhysics(force)
        x, x_dot, theta, theta_dot = self.state

        # Check termination conditions
        done = (
            x < -self.x_threshold
            or x > self.x_threshold
            or theta < -self.theta_threshold_radians
            or theta > self.theta_threshold_radians
        )
        done = bool(done)

        # Reward logic
        if not done:
            reward = 1.0
        elif self.steps_beyond_done is None:
            # Pole just fell!
            self.steps_beyond_done = 0
            reward = 1.0
        else:
            if self.steps_beyond_done == 0:
                logger.warn(
                    """
    You are calling 'step()' even though this environment has already returned
    done = True. You should always call 'reset()' once you receive 'done = True'
    Any further steps are undefined behavior.
                    """
                )
            self.steps_beyond_done += 1
            reward = 0.0

        # Optionally add noise to the returned observation
        obs = np.array(self.state, dtype=np.float32)
        if self.obs_noise_scale > 0.0:
            obs += self.np_random.normal(0.0, self.obs_noise_scale, size=4).astype(np.float32)

        return obs, reward, done, {}

    def reset(self):
        # Random initial state
        self.state = self.np_random.uniform(low=-0.05, high=0.05, size=(4,))
        self.steps_beyond_done = None
        return np.array(self.state, dtype=np.float32)

    def render(self, mode="human"):
        screen_width = 600
        screen_height = 400

        world_width = self.x_threshold * 2
        scale = screen_width / world_width
        carty = 100  # TOP OF CART
        polewidth = 10.0
        polelen = scale * 1.0
        cartwidth = 50.0
        cartheight = 30.0

        if self.viewer is None:
            from gym.envs.classic_control import rendering

            self.viewer = rendering.Viewer(screen_width, screen_height)
            l, r, t, b = (
                -cartwidth / 2,
                cartwidth / 2,
                cartheight / 2,
                -cartheight / 2,
            )
            axleoffset = cartheight / 4.0
            cart = rendering.FilledPolygon([(l, b), (l, t), (r, t), (r, b)])
            self.carttrans = rendering.Transform()
            cart.add_attr(self.carttrans)
            self.viewer.add_geom(cart)
            l, r, t, b = (
                -polewidth / 2,
                polewidth / 2,
                polelen - polewidth / 2,
                -polewidth / 2,
            )
            pole = rendering.FilledPolygon([(l, b), (l, t), (r, t), (r, b)])
            pole.set_color(0.8, 0.6, 0.4)
            self.poletrans = rendering.Transform(translation=(0, axleoffset))
            pole.add_attr(self.poletrans)
            pole.add_attr(self.carttrans)
            self.viewer.add_geom(pole)
            self.axle = rendering.make_circle(polewidth / 2)
            self.axle.add_attr(self.poletrans)
            self.axle.add_attr(self.carttrans)
            self.axle.set_color(0.5, 0.5, 0.8)
            self.viewer.add_geom(self.axle)
            self.track = rendering.Line((0, carty), (screen_width, carty))
            self.track.set_color(0, 0, 0)
            self.viewer.add_geom(self.track)

        if self.state is None:
            return None

        x = self.state
        cartx = x[0] * scale + screen_width / 2.0  # MIDDLE OF CART
        self.carttrans.set_translation(cartx, carty)
        self.poletrans.set_rotation(-x[2])

        return self.viewer.render(return_rgb_array=(mode == "rgb_array"))

    def close(self):
        if self.viewer:
            self.viewer.close()
            self.viewer = None


# ---------------------------------------------------
# Register the stochastic environment with Gym
# ---------------------------------------------------
from gym.envs.registration import register

register(
    id="Continuous-CartPole-v0",
    entry_point="Continuous_CartPole:ContinuousCartPoleEnv",
    max_episode_steps=200,
)
